<?php

namespace xo\access;

class AccessService {
	private static $PASSWORD_MIN_LENGTH = 8;
	private static $PASSWORD_MAX_LENGTH = 12;
	private static $EMAIL_MAX_LENGTH = 150;
	private static $LOGIN_MAX_LENGTH = 150;
	private static $CURRENT_ACCESS_SESSION_KEY = 'CURRENT_ACCESS';
	private $db;
	public function __construct($db) {
		$this->db = $db;
	}
	public function generateRandomPassword() {
		return LoginPasswordUtils::generateRandomPassword ( self::$PASSWORD_MIN_LENGTH, self::$PASSWORD_MAX_LENGTH );
	}
	public function isAnEmail($email) {
		return (preg_match ( "/^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z]+$/", $email ) == 1);
	}
	public function accessAlreadyExistForLogin($login) {
		$access = null;
		$query = "select count(*) as count from ACCESS_ACCESS where login=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$login 
		) );
		$row = $stmt->fetch ();
		return $row ['count'] > 0;
	}
	public function createAccess($email, $login = null, $password) {
		if ($login == null) {
			$login = $email;
		}
		if ($this->accessAlreadyExistForLogin ( $login )) {
			throw new ServiceFunctionalException ( "Un accès a déjà été créé pour le login : " . $login );
		}
		if (strlen ( $password ) < self::$PASSWORD_MIN_LENGTH) {
			throw new ServiceFunctionalException ( "Le mot de passe doit contenir au moins " . self::$PASSWORD_MIN_LENGTH . " signes." );
		}
		if (strlen ( $email ) > self::$EMAIL_MAX_LENGTH) {
			throw new ServiceFunctionalException ( "L'email ne doit pas contenir plus de " . self::$EMAIL_MAX_LENGTH . " signes." );
		}
		if (! $this->isAnEmail ( $email )) {
			throw new ServiceFunctionalException ( "L'email n'est pas valide." );
		}
		if (strlen ( $login ) > self::$LOGIN_MAX_LENGTH) {
			throw new ServiceFunctionalException ( "Le login ne doit pas contenir plus de " . self::$LOGIN_MAX_LENGTH . " signes." );
		}
		
		$utils = new LoginPasswordUtils ( $password );
		try {
			$query = "insert into ACCESS_ACCESS(email,login,password,salt,status) values(?,?,?,?,?)";
			$stmt = $this->db->prepare ( $query );
			$stmt->execute ( array (
					$email,
					$login,
					$utils->getHash (),
					$utils->getSalt (),
					Access::$STATUS_ACTIVE 
			) );
			return $this->db->lastInsertId ();
		} catch ( Exception $e ) {
			error_log ( $e->getMessage () );
			throw new ServiceFunctionalException ( "Une erreur est survenue lors de la création de l'idée." );
		}
	}
	public function updateAccessStatus($accessId, $status) {
		$query = "update ACCESS_ACCESS set status=? where id=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$status,
				$accessId 
		) );
	}
	public function updateAccessPassword($accessId, $password) {
		if (strlen ( $password ) < self::$PASSWORD_MIN_LENGTH) {
			throw new ServiceFunctionalException ( "Le mot de passe doit contenir au moins " . self::$PASSWORD_MIN_LENGTH . " signes." );
		}
		
		$utils = new LoginPasswordUtils ( $password );
		$query = "update ACCESS_ACCESS set password=?,salt=? where id=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$utils->getHash (),
				$utils->getSalt (),
				$accessId 
		) );
	}
	public function login($login, $password) {
		$access = null;
		$query = "select * from ACCESS_ACCESS where login=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$login 
		) );
		$row = $stmt->fetch ();
		
		if ($row) {
			$utils = new LoginPasswordUtils ( $password, $row ['salt'] );
			
			if ($utils->getHash () == $row ['password']) {
				$access = new Access ( $row ['id'], $row ['email'], $row ['login'] );
				$access->setStatus ( $row ['status'] );
			}
		}
		return $access;
	}
	public function hasAuthorization($accessId, $authToken) {
		$query = "select count(*) as count from ACCESS_ACCESS a
		join ACCESS_ACCESS_ROLE_MAP m on a.id=m.access_id
		join ACCESS_ROLE_AUTH_TOKEN_MAP tm on tm.role_id=m.role_id
		join ACCESS_AUTH_TOKEN t on t.id=tm.token_id
		where a.id=? and t.code=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$accessId,
				$authToken 
		) );
		$row = $stmt->fetch ();
		return $row ['count'] > 0;
	}
	public function getRoles() {
		$query = "select * from ACCESS_ROLE order by code";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array () );
		$rows = $stmt->fetchAll ();
		$roles = array ();
		foreach ( $rows as $row ) {
			$role = new Role ( $row ['id'], $row ['code'] );
			$role->setLabel ( $row ['label'] );
			array_push ( $roles, $role );
		}
		return $roles;
	}
	public function getRole($roleId) {
		$query = "select * from ACCESS_ROLE where id=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$roleId 
		) );
		$row = $stmt->fetch ();
		$role = new Role ( $row ['id'], $row ['code'] );
		$role->setLabel ( $row ['label'] );
		return $role;
	}
	public function getAccessRoles($accessId) {
		$query = "select r.* from ACCESS_ROLE r join ACCESS_ACCESS_ROLE_MAP arm on r.id=arm.role_id where arm.access_id=? order by r.code";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$accessId 
		) );
		$rows = $stmt->fetchAll ();
		$roles = array ();
		foreach ( $rows as $row ) {
			$role = new Role ( $row ['id'], $row ['code'] );
			$role->setLabel ( $row ['label'] );
			array_push ( $roles, $role );
		}
		return $roles;
	}
	public function setAccessRoles($accessId, $roleIds) {
		$query = "delete from ACCESS_ACCESS_ROLE_MAP  where access_id=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$accessId 
		) );
		if (count ( $roleIds ) > 0) {
			$query = "insert into ACCESS_ACCESS_ROLE_MAP(access_id,role_id) values ";
			$data = array ();
			foreach ( $roleIds as $roleId ) {
				$query .= '(?,?),';
				array_push ( $data, $accessId );
				array_push ( $data, $roleId );
			}
			$query = substr ( $query, 0, strlen ( $query ) - 1 );
			$stmt = $this->db->prepare ( $query );
			$stmt->execute ( $data );
		}
	}
	public function getAccess($accessId) {
		$query = "select * from ACCESS_ACCESS where id=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$accessId 
		) );
		$row = $stmt->fetch ();
		$access = null;
		if ($row) {
			$access = new Access ( $row ['id'], $row ['email'], $row ['login'] );
			$access->setStatus ( $row ['status'] );
		}
		
		return $access;
	}
	public function getAccessFromLogin($login) {
		$query = "select * from ACCESS_ACCESS where login=?";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$login 
		) );
		$row = $stmt->fetch ();
		$access = null;
		if ($row) {
			$access = new Access ( $row ['id'], $row ['email'], $row ['login'] );
			$access->setStatus ( $row ['status'] );
		}
		
		return $access;
	}
	public function getAccessFromEmail($email) {
	}
	public function getAccesses() {
		$query = "select * from ACCESS_ACCESS order by login";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array () );
		$accesses = array ();
		$rows = $stmt->fetchAll ();
		foreach ( $rows as $row ) {
			$access = new Access ( $row ['id'], $row ['email'], $row ['login'] );
			$access->setStatus ( $row ['status'] );
			array_push ( $accesses, $access );
		}
		return $accesses;
	}
	public function getRoleAuthorizationTokens($roleId) {
		$query = "select at.* from ACCESS_AUTH_TOKEN at join ACCESS_ROLE_AUTH_TOKEN_MAP atrm on at.id=atrm.token_id where atrm.role_id = ? order by code";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array (
				$roleId 
		) );
		$tokens = array ();
		$rows = $stmt->fetchAll ();
		foreach ( $rows as $row ) {
			$token = new AuthorizationToken ( $row ['id'], $row ['code'] );
			$token->setDescription ( $row ['description'] );
			array_push ( $tokens, $token );
		}
		return $tokens;
	}
	public function getAccessAuthorizationTokens() {
		$query = "select * from ACCESS_AUTHORIZATION_TOKEN order by code";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array () );
		$tokens = array ();
		$rows = $stmt->fetchAll ();
		foreach ( $rows as $row ) {
			$token = new AuthorizationToken ( $row ['id'], $row ['code'] );
			$token->setDescription ( $row ['description'] );
			array_push ( $tokens, $token );
		}
		return $tokens;
	}
	public function getAuthorizationTokens() {
		$query = "select * from ACCESS_AUTH_TOKEN order by code";
		$stmt = $this->db->prepare ( $query );
		$stmt->execute ( array () );
		$tokens = array ();
		$rows = $stmt->fetchAll ();
		foreach ( $rows as $row ) {
			$token = new AuthorizationToken ( $row ['id'], $row ['code'] );
			$token->setDescription ( $row ['description'] );
			array_push ( $tokens, $token );
		}
		return $tokens;
	}
	
	// SESSION
	public static function getCurrentAccessId() {
		$currentAccessId = null;
		if (isset ( $_SESSION [self::$CURRENT_ACCESS_SESSION_KEY] )) {
			$currentAccessId = $_SESSION [self::$CURRENT_ACCESS_SESSION_KEY];
		}
		return $currentAccessId;
	}
	public static function setCurrentAccess(Access $access) {
		$_SESSION [self::$CURRENT_ACCESS_SESSION_KEY] = $access->getId ();
	}
	public static function removeCurrentAccess() {
		unset ( $_SESSION [self::$CURRENT_ACCESS_SESSION_KEY] );
	}
	public function getCurrentAccess() {
		$currentAccessId = self::getCurrentAccessId ();
		return $currentAccessId == null ? null : $this->getAccess ( $currentAccessId );
	}
}

?>
